const char rebase_license[] =
  "\trebase - A math game\n"
  "\tCopyright (C) 2009 Pedro Izecksohn\n\n"
  "\tThis program is free, so it has no warranty at all.\n"
  "\tTo redistribute this program:\n"
  "\tYou must make sure the source code is available to the receiver;\n"
  "\tYou must not take the authorship from me;\n"
  "\tYou must not modify this license.\n";

#include <assert.h>
#include <errno.h>
#include <time.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

typedef struct
{
  FILE *f;
  unsigned long long int sum_of_seconds;
  unsigned int nresults;
} statistics;

void statistics_init (statistics * st, const char *filename);
void statistics_append (statistics * st, int value, int base,
			unsigned long long int took_seconds);

int
main (int argc, char **argv)
{
  const char help[] =
    "Optional argument:\n"
    "--valuemax integer\t- Sets valuemax to the desired integer.\n"
    "The default valuemax is 0x7f.\n";

  if ((argc > 1) && (argc != 3))
    {
      printf ("%s\n%s", rebase_license, help);
      exit (EXIT_SUCCESS);
    }

  int valuemax = 0x7F;
  if (3 == argc)
    {
      if (!strcmp (argv[1], "--valuemax"))
	valuemax = strtol (argv[2], NULL, 0);
      else
	{
	  fprintf (stderr, "Unrecognized argument %s\n", argv[1]);
	  return EXIT_FAILURE;
	}
    }

#ifndef MAX_PLAY_SECONDS
#define MAX_PLAY_SECONDS 4500
#endif

  time_t time_playing;
  const time_t began_to_play = time (NULL);
  srand (began_to_play);

  statistics st;
  statistics_init (&st, "rebase_stats.txt");

  printf ("To exit press Control c\n");
  do
    {
      int base;
      do
	{
	  base = ((rand () % 35) + 2);
	}
      while (10 == base);

      int value = (rand () % valuemax);

      char buffer[34];
      int entered;
      time_t began = time (NULL);

      do
	{
	  printf ("Convert %d decimal to base %d: ", value, base);

	  char *fgets_returned = fgets (buffer, sizeof buffer, stdin);
	  if (!fgets_returned)
	    {
	      if (ferror (stdin))
		{
		  perror ("fgets");
		  return EXIT_FAILURE;
		}
	      return EXIT_SUCCESS;
	    }

	  entered = strtol (buffer, NULL, base);
	  if (entered == value)
	    break;
	  printf ("It is wrong.\n");

	}
      while (1);

      const time_t took_seconds = (time (NULL) - began);
      printf ("You took %lu seconds to calculate it right.\n", took_seconds);

      if (st.nresults > 1)
	printf ("Your average was %llu seconds.\n",
		st.sum_of_seconds / st.nresults);
      statistics_append (&st, value, base, took_seconds);

      time_playing = (time (NULL) - began_to_play);
    }
  while (time_playing <= MAX_PLAY_SECONDS);

  printf ("You are playing for more than %lu minutes.\nDo something else.\n",
	  (time_playing / 60));
  getchar ();
  return EXIT_SUCCESS;
}

void
statistics_init (statistics * st, const char *filename)
{
  assert (st);
  st->f = fopen (filename, "rb+");
  if ((!st->f) && (ENOENT == errno))
    st->f = fopen (filename, "wb+");
  if (!st->f)
    {
      fprintf (stderr, "%s could not open nor create %s\n%s\n",
	       __func__, filename, strerror (errno));
      abort ();
    }

  st->sum_of_seconds = 0;
  st->nresults = 0;
  int fscanf_returned = 0;
  do
    {
      unsigned long long int seconds;
      fscanf_returned = fscanf (st->f, "%*d\t%*u\t%llu\n", &seconds);
      if (1 == fscanf_returned)
	{
	  st->sum_of_seconds += seconds;
	  ++(st->nresults);
	}
      else
	break;
    }
  while (1);
  if (ferror (st->f))
    {
      fprintf (stderr, "An error occurred while I was reading %s\n%s\n",
	       filename, strerror (errno));
      abort ();
    }
  if ((fscanf_returned < 1) && (!feof (st->f)))
    {
      fprintf (stderr, "Unrecognized format of %s\n", filename);
      abort ();
    }

  if (fseek (st->f, 0, SEEK_END))
    {
      fprintf (stderr, "%s could not fseek.\n%s\n", __func__,
	       strerror (errno));
      abort ();
    }
}

void
statistics_append (statistics * st, int value, int base,
		   unsigned long long int took_seconds)
{
  int fprintf_returned =
    fprintf (st->f, "%d\t%d\t%llu\n", value, base, took_seconds);
  if (fprintf_returned < 0)
    {
      fprintf (stderr, "%s could not fprintf.\n%s\n", __func__,
	       strerror (errno));
      abort ();
    }

  if (fflush (st->f))
    {
      fprintf (stderr, "%s could not fflush.\n%s\n", __func__,
	       strerror (errno));
      abort ();
    }

  st->sum_of_seconds += took_seconds;
  ++(st->nresults);
}
