#include <math.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

typedef struct {
  double *pv;
  double *fv;
  double *nperiods;
  double *rate;
} loan;

unsigned int count_filled (loan *l)
{
  unsigned int ret = 0;
  if (l->pv) ++ret;
  if (l->fv) ++ret;
  if (l->nperiods) ++ret;
  if (l->rate) ++ret;
  return ret;
}

#define bufsize 1024
char buffer [bufsize];
#define clean_buffer() memset(buffer, 0, bufsize)

void ask_double (const char *const message, double *const d, double **pd)
{
  *pd = NULL;
  ask_again:
  printf ("%s", message);
  clean_buffer();
  if (!fgets (buffer, bufsize, stdin)) abort();
  if (1 < strlen (buffer))
  {
    char *endptr;
    *d = strtod (buffer, &endptr);
    if ((*endptr != '\n') && (*endptr != '%'))
    {
      printf ("I can not understand what you just entered.\n");
      goto ask_again;
    }
    *pd = d;
  }
}

/**
Returns the compound interest rate.
*/
double calculate_rate (const double pv, const double fv, const double nperiods)
{
  // fprintf (stderr, "I\'m inside %s (%.lf, %.lf, %.lf\n", __func__, pv, fv, nperiods);
  double rate = (fv/pv);
  if (rate == 1.0) return 1.0;
  if (0.0 == nperiods) return NAN;
  if (rate > 1.0)
  {
    rate /= nperiods;
    if (rate <= 1.0) rate += 1.0;
  }
  else rate = (1.0-(rate/nperiods));
  double rate_diff = 64;
  double current_rate;
  enum sign {notsigned, positive, negative} previous = notsigned;
  do
  {
    current_rate = rate;
    double calculated_fv = (pv * pow (rate, nperiods));
    // fprintf (stderr, "fv = %la\trate = %lf\tcalculated_fv = %la\trate_diff = %la\n",
    //                   fv,       rate,       calculated_fv,       rate_diff);
    if (fv == calculated_fv)
    {
      // fprintf (stderr, "fv == calculated_fv\n");
      break;
    }
    else if (calculated_fv > fv)
    {
      // fprintf (stderr, "calculated_fv > fv: %la > %la\n", calculated_fv, fv);
      if (negative == previous) rate_diff *= 2;
      previous = positive;
      rate -= ((rate-((fv>pv)?1:0))/rate_diff);
    }
    else // (calculated_fv < fv)
    {
      // fprintf (stderr, "calculated_fv < fv: %la < %la\n", calculated_fv, fv);
      if (positive == previous) rate_diff *= 2;
      previous = negative;
      rate += ((rate-((fv>pv)?1:0))/rate_diff);
    }
    // gcc -O1 would work if you'd uncomment the following line:
    // fprintf (stderr, "current_rate %c= rate\trate = %a\n", ((current_rate == rate)?'=':'!'), rate);
  } while (current_rate != rate);
  return rate;
}

int main ()
{
  printf (
"\n"
"Author: Pedro Izecksohn\n"
"License: This program is distributable only if its sources and its author`s\n"
"name be included in the package.\n"
"\n"
"Answer what you have.\n"
"Leave empty what you do not have.\n"
"\n"
"Press enter after each answer.\n"
"To leave a field empty just press enter.\n"
"\n"
  );

  double pv, fv, nperiods, rate;
  loan l = {NULL, NULL, NULL, NULL};

  for (;;)
  {

    if (!(l.pv))
    {
      ask_double ("Present value: ", &pv, &l.pv);
      if (l.pv && ( pv < 0.0 ) )
      {
        l.pv = NULL;
        printf ("The present value can not be negative.\n");
        continue;
      }
    }

    if (!(l.fv)) ask_double ("Future value: ", &fv, &l.fv);

    ask_nperiods:
    if (!(l.nperiods))
    {
      ask_double ("Number of periods: ", &nperiods, &l.nperiods);
      if (l.nperiods && ( nperiods < 0.0 ) )
      {
        l.nperiods = NULL;
        printf ("The number of periods can not be negative.\n");
        goto ask_nperiods;
      }
    }

    ask_rate:
    if (count_filled (&l) < 3)
    {
      ask_double ("For percentage append %\nRate: ", &rate, &l.rate);
      if (l.rate)
      {
        if (strchr (buffer, '%'))
        {
          rate /= 100;
          rate += 1 ;
        }
        if ( rate < 0.0 )
        {
          l.rate = NULL;
          printf (
            "Multiplier rate: %lf\n"
            "A multiplier rate can not be negative.\n"
            "To use percentage put %% after the number.\n",
            rate
          );
          goto ask_rate;
        }
      }
    }

    if (count_filled (&l) > 2) break;
    else printf ("Less than 3 fields were filled.\n");
  }

  if (!(l.pv))
  {
    const double total_rate = pow (rate, nperiods);
    pv = ( fv / total_rate );
    printf ("Present value: %.3lf", pv);
    goto end;
  }

  if (!(l.fv)) // Apply rate
  {
    // fprintf (stderr, "pv: %.lf\nnperiods: %.lf\n", pv, nperiods);
    printf ("Future value: %.4lf", (pv * pow (rate, nperiods)) );
    goto end;
  }

  if (!(l.nperiods))
  {
    if ( fv == pv  )
    {
      if ( 1.0 == rate ) printf ("Number of periods: Indeterminable");
      else printf ("You entered some wrong value.");
      goto end;
    }
    // fv != pv
    if ( 0.0 == pv )
    {
      printf ("It is impossible.");
      goto end;
    }
    if   ( ( rate == 1.0 ) ||
         ( ( rate  < 1.0 ) && (fv > pv) ) ||
         ( ( rate  > 1.0 ) && (fv < pv) ) )
    {
      printf ("Number of periods: Never");
      goto end;
    }
    const double total_rate = ( fv / pv );
    double calculated_total_rate;
    double increment_period_by = 1;
    nperiods = 0;
    do
    {
      nperiods += increment_period_by;
      calculated_total_rate = pow (rate, nperiods);
      if ( ( ( total_rate > 1.0 ) && ( calculated_total_rate > total_rate ) ) ||
           ( ( total_rate < 1.0 ) && ( calculated_total_rate < total_rate ) ) )
      {
        nperiods -= increment_period_by;
        increment_period_by /= 2;
      }
    } while ( ( calculated_total_rate != total_rate ) && ( 0.0 != increment_period_by ) );
    printf ("Number of periods: %.3lf", nperiods);
    goto end;
  }

  if (!(l.rate))
  {
    rate = calculate_rate (pv, fv, nperiods);
    const double percentage = ( ( ((rate>=1.0)||isnan(rate)) ? (rate-1.0) : -(1.0-rate) ) * 100 );
    printf ("Rate: %.8lf%%", percentage);
  }

  end:
  getchar ();
  return 0;
}
